<?php
/**
 * UsuarioDao.php
 * Modelo para la persistencia de datos de usuarios
 * @copyright (c) 2016, Merdiz
 * @author José Francisco Montaño Andriano
 * @version 1.0 Beta
 * @package usuarios
 */

require_once BASEPATH . 'core' . DIRECTORY_SEPARATOR . 'Model.php';
require_once BASEPATH . 'application' . DIRECTORY_SEPARATOR . 'usuarios' .
    DIRECTORY_SEPARATOR . 'models' . DIRECTORY_SEPARATOR . 'Usuario.php';
require_once BASEPATH . 'helpers' . DIRECTORY_SEPARATOR . 'Utils.php';

class UsuarioDao extends Model
{

    /**
     * Método constructor, obtiene la conexión a la base de datos
     * @access public
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Método para validar si existe usuario
     * @param Usuario $u
     * @return mixed array con los datos de respuesta
     */
    public function existeUsuario(Usuario $u)
    {
        $sql = "SELECT `Id_Usuario` FROM `usuarios` WHERE `Usuario` = ? and Status = 1";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $u->__get('_usuario'), PDO::PARAM_STR);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['datos'] = $stmt->rowCount();
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), __FILE__, __LINE__);
            $response['status'] = false;
            $response['error'] = 'Lo siento, no pude validar los datos de usuario. '
                . TICKETS;
        }

        return $response;
    }

    /**
     * Método para validar si el usuario es activo
     * @param Usuario $u
     * @return mixed array con los datos de respuesta
     */
    public function esUsuarioActivo(Usuario $u)
    {
        $sql = "SELECT `Status` FROM `usuarios` WHERE `Usuario` = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $u->__get('_usuario'), PDO::PARAM_STR);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['datos'] = $stmt->fetch(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), __FILE__, __LINE__);
            $response['status'] = false;
            $response['error'] = 'Lo siento, no pude validar los datos de usuario. '
                . TICKETS;
        }

        return $response;
    }

    /**
     * Método para obtener los datos de inicio de sesión
     * @param Usuario $u
     * @return mixed array con los datos de respuesta
     */
    public function login(Usuario $u)
    {
        $sql = "SELECT `Id_Usuario`, `Usuario`, `Password`, `Id_Sucursal`, `Tipo` FROM "
            . "`usuarios` WHERE `Usuario` = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $u->__get('_usuario'), PDO::PARAM_STR);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['login'] = $stmt->fetch(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), __FILE__, __LINE__);
            $response['status'] = false;
            $response['error'] = 'Lo siento, no pude obtener los datos de acceso. '
                . TICKETS;
        }

        return $response;
    }

    public function obtenerUsuariosPorGerente($idGerente)
    {
        $sql = "SELECT * FROM `usuarios` WHERE `Id_Gerente` = ? AND `Status` = 1";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idGerente, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, no pude obtener los datos de los usuarios. '
                . TICKETS;
        }

        return $response;
    }

    public function obtenerUsuario($idUsuario)
    {
        $sql = "SELECT * FROM `usuarios` WHERE `Id_Usuario` = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idUsuario, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetch(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, no pude obtener los datos del usuario. '
                . TICKETS;
        }

        return $response;
    }

    public function getVencimientoLicencia($fechaSemana){
        $sql = "SELECT * FROM `usuarios` WHERE `Vigencia_Licencia` = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $fechaSemana, PDO::PARAM_STR);
        try{
            $stmt->execute();
            $response['status'] = true;
            $response['licencia'] = $stmt->fetch(PDO::FETCH_OBJ);
            $response['total'] = $this->_dbh->query('SELECT FOUND_ROWS();')->fetch(PDO::FETCH_COLUMN);
        }
        catch(PDOException $e){
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener fecha de vencimiento de la licencia';
        }

        return $response;
    }



    public function obtenerUsuarioPorCodigoBarras($codigoBarras)
    {
        $sql = "SELECT * FROM `usuarios` WHERE `Codigo_Barras` = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $codigoBarras, PDO::PARAM_STR);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['total'] = $stmt->rowCount();
            $response['usuarios'] = $stmt->fetch(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Ocurrió un error al obtener los datos de usuario. ' . TICKETS;
        }

        return $response;
    }

    public function obtenerUsuariosConMando()
    {
        $sql = "SELECT * FROM `usuarios` WHERE `Mando` = 1 AND `Status` = 1";
        $stmt = $this->_dbh->prepare($sql);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos de usuarios';
        }

        return $response;
    }

    public function obtenerUsuariosSinMando()
    {
        $sql = "SELECT * FROM `usuarios` WHERE `Mando` = 0 AND `Status` = 1";
        $stmt = $this->_dbh->prepare($sql);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos de usuarios';
        }

        return $response;
    }

    public function obtenerUsuariosActivos($idSucursal)
    {
        $sql = "SELECT * FROM `usuarios` WHERE `Status` = 1 AND `Id_Sucursal` = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idSucursal, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos de usuarios';
        }

        return $response;
    }
    
      public function obtenerUsuariosActivosNoJefes($idSucursal)
    {
        $sql = "SELECT * FROM `usuarios` WHERE `Status` = 1 AND `Id_Sucursal` = ? AND `Id_Usuario` != 16 AND `Id_Usuario` != 17";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idSucursal, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos de usuarios';
        }

        return $response;
    }
    
       public function obtenerUsuariosActivosSinJefes()
    {
        $sql = "SELECT * FROM `usuarios` WHERE `Status` = 1 AND `Id_Usuario` != 16 AND `Id_Usuario` != 17";
        $stmt = $this->_dbh->prepare($sql);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos de usuarios';
        }

        return $response;
    }

    public function obtenerTodosUsuariosActivos()
    {
        $sql = "SELECT * FROM `usuarios` WHERE `Status` = 1";
        $stmt = $this->_dbh->prepare($sql);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos de usuarios';
        }

        return $response;
    }

    public function obtenerTodosUsuariosActivosNoConformidad()
    {
        $sql = "SELECT * FROM `usuarios` WHERE `Status` >= 1 AND `Id_Usuario` != 16 AND `Id_Usuario` != 17";
        $stmt = $this->_dbh->prepare($sql);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos de usuarios';
        }

        return $response;
    }


    public function obtenerUsuariosActivosForSucursal($idSucursal, $tipo)
    {
        $sql = "SELECT * FROM `usuarios` WHERE `Status` >= 1 AND `Id_Sucursal` = ? AND `Tipo` >= ? and Id_Usuario != 28 and Id_Usuario != 27 and Id_Usuario != 29 and Id_Usuario != 122 ";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idSucursal, PDO::PARAM_INT);
        $stmt->bindValue(2, $tipo, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos de usuarios';
        }

        return $response;
    }


    public function obtenerTodosUsuariosActivosSinDirectores()
    {
        $sql = "SELECT * FROM `usuarios` WHERE `Status` = 1 AND `Id_Usuario` <> 16 AND `Id_Usuario` <> 17";
        $stmt = $this->_dbh->prepare($sql);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos de usuarios';
        }

        return $response;
    }

    public function obtenerUsuariosTelemarketing()
    {
        $sql = "SELECT `Id_Usuario`, `Nombre`, `Apellido_Paterno`, `Apellido_Materno` FROM "
            . "`usuarios` WHERE `Marketing` = 1 AND `Status` = 1";
        $stmt = $this->_dbh->prepare($sql);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los usuarios '
                . 'de Telemarkting. ' . TICKETS;
        }

        return $response;
    }

    public function obtenerUsuariosTelemarketingSucursal($idSucursal)
    {
        $sql = "SELECT `Id_Usuario`, `Nombre`, `Apellido_Paterno`, `Apellido_Materno` FROM "
            . "`usuarios` WHERE `Marketing` = 1 AND `Status` > 0 AND Id_Sucursal = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idSucursal, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los usuarios '
                . 'de Telemarkting. ' . TICKETS;
        }

        return $response;
    }

    public function obtenerGerenteUsuario($idUsuario)
    {
        $sql = "SELECT `Id_Gerente` FROM `usuarios` WHERE `Id_Usuario` = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idUsuario, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['gerente'] = $stmt->fetch(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos. '
                . TICKETS;
        }

        return $response;
    }

    public function obtenerRol($idUsuario)
    {
        $sql = "SELECT `Tipo` FROM `usuarios` WHERE `Id_Usuario` = $idUsuario";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idUsuario, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['rol'] = $stmt->fetch(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener tu rol de usuario. '
                . TICKETS;
        }

        return $response;
    }

    public function obtenerVendedoresActivos()
    {
        $sql = "SELECT * FROM `usuarios` WHERE (`Tipo` = 4 OR `Tipo` = 2) AND `Status` >= 1";
        $stmt = $this->_dbh->prepare($sql);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['vendedores'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener la lista '
                . 'de vendedores. ' . TICKETS;
        }

        return $response;
    }

    public function obtenerVendedoresActivosCron()
    {
        $sql = "SELECT * FROM `usuarios` WHERE (`Tipo` = 4 OR `Tipo` = 2) AND `Status` >= 1 AND id_Usuario <> 63 AND id_Usuario <> 130";
        $stmt = $this->_dbh->prepare($sql);
        try{
            $stmt->execute();
            $response['status'] = true;
            $response['vendedores'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener la lista '
                . 'de vendedores. ' . TICKETS;
        }

        return $response;
    }

    public function obtenerVendedoresActivosPorSucursal($idSucursal)
    {
        $sql = "SELECT * FROM `usuarios` WHERE (`Tipo` = 4 OR `Tipo` = 2) AND `Status` = 1 AND `Id_Sucursal` = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idSucursal, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['vendedores'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener la lista '
                . 'de vendedores. ' . TICKETS;
        }

        return $response;
    }

    public function obtenerVendedoresActivosPorSucursalComisiones($idSucursal)
    {
        $sql = "SELECT * FROM `usuarios` WHERE `Tipo` = 4 AND `Status` = 1 AND `Id_Sucursal` = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idSucursal, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['vendedores'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener la lista '
                . 'de vendedores. ' . TICKETS;
        }

        return $response;
    }

    public function obtenerDatosNomina($idNomina)
    {
        $sql = "SELECT `usuarios`.`Nombre`, `usuarios`.`Apellido_Paterno`, `usuarios`.`Apellido_Materno`, "
            . "FROM `nomina_pago` INNER JOIN `usuarios` ON `usuarios`.`Id_Usuario` = `nomina_pago`.`Id_Usuario` "
            . "WHERE `nomina_pago`.`Id_Nomina` = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idNomina, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['datosNomina'] = $stmt->fetch(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos '
                . 'de la nomina. ' . TICKETS;
        }

        return $response;
    }

    public function validarAsistencia($idUsuario, $idSucursal, $fecha)
    {
        $sql = "SELECT `Id_Usuario` FROM `asistencia` WHERE `Id_Usuario` = ? AND `Id_Sucursal` = ? "
            . "AND `Fecha` = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idUsuario, PDO::PARAM_INT);
        $stmt->bindValue(2, $idSucursal, PDO::PARAM_INT);
        $stmt->bindValue(3, $fecha, PDO::PARAM_STR);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['total'] = $stmt->rowCount();
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un ';
        }
    }

    public function actualizarContrasena(Usuario $u)
    {
        $sql = "UPDATE `usuarios` SET `Password` = ?, `fechaPassword` = ?, `passwordActualizada` = ? WHERE `Id_Usuario` = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $u->__get('_contrasena'), PDO::PARAM_STR);
        $stmt->bindValue(2, $u->__get('_fechaPassword'), PDO::PARAM_STR);
        $stmt->bindValue(3, 1, PDO::PARAM_INT);
        $stmt->bindValue(4, $u->__get('_idUsuario'), PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al actualizar '
                . 'tu contraseña. ' . TICKETS;
        }

        return $response;
    }

    public function validarEmailUsuario($email)
    {
        $sql = "SELECT `Id_Usuario` FROM `usuarios` WHERE `Mail` = ? AND `Status` = 1";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $email, PDO::PARAM_STR);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['valido'] = $stmt->rowCount();
            $response['datosUsuario'] = $stmt->fetch(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al validar el Email';
        }

        return $response;
    }

    public function obtenerUsuariosPorRol($rol)
    {
        $sql = "SELECT * FROM `usuarios` WHERE `Tipo` = ? AND `Status` = 1";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $rol, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Ocurrió un error al obtener los usuarios por tipo. ' . TICKETS;
        }

        return $response;
    }

    public function cambiaStatus($idUsuario, $status)
    {
        $sql = "UPDATE `usuarios` SET `Status` = ? WHERE `Id_Usuario` = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $status, PDO::PARAM_INT);
        $stmt->bindValue(2, $idUsuario, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Ocurrió un error al cambiar el status del usuario. ' . TICKETS;
        }

        return $response;
    }

    /**  CRUD USUARIOS  **/
    public function obtenerUsuariosSucursal($idSucursal, $inicio, $registros)
    {
        $sql = "SELECT * FROM `usuarios` WHERE `Id_Sucursal` = ? ORDER BY Status DESC LIMIT ?, ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idSucursal, PDO::PARAM_INT);
        $stmt->bindValue(2, $inicio, PDO::PARAM_INT);
        $stmt->bindValue(3, $registros, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos de usuarios';
        }

        return $response;
    }

    public function totalUsuariosSucursal($idSucursal)
    {
        $sql = "SELECT COUNT(`Id_Usuario`) FROM `usuarios` WHERE `Id_Sucursal` = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idSucursal, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['total'] = $stmt->fetchColumn();
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento no pude validar el total de usuarios de la '
                . 'sucursal. ' . TICKETS;
        }

        return $response;
    }

    /*
     * Metodo totalCompletarUsuarioDAO
     * Param: Recibe desde 1 letra hasta el usuario completo
     * Desarrollo: AGRA
     * Descripción: el metodo regresara el total de usuarios y todos los usuarios que coincidan con la busqueda
     * por el momento sin distinción de sucursal
     */

    public function totalCompletarUsuarioDAO($usuario, $inicio, $registros)
    {

        $sql = "SELECT SQL_CALC_FOUND_ROWS * FROM `usuarios` WHERE Usuario LIKE ? 
                        OR CONCAT(Nombre,' ',Apellido_Paterno,' ',Apellido_Materno) LIKE ? 
                        OR Mail LIKE ? 
                        LIMIT ?, ?";
        $stmt = $this->_dbh->prepare($sql);
        //$stmt->bindValue(1, $usuario, PDO::PARAM_STR );
        $stmt->bindValue(1, "%$usuario%", PDO::PARAM_STR);
        $stmt->bindValue(2, "%$usuario%", PDO::PARAM_STR);
        $stmt->bindValue(3, "%$usuario%", PDO::PARAM_STR);
        $stmt->bindValue(4, $inicio, PDO::PARAM_INT);
        $stmt->bindValue(5, $registros, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
            $response['total'] = $this->_dbh->query('SELECT FOUND_ROWS();')->fetch(PDO::FETCH_COLUMN);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al buscar los usuarios';
        }

        return $response;

    }

    /*
     * Método: generaNoEmpleadoDAO
     * Descripción:  obtiene nuevo número de empleado
     * Param:  ano, mes, dia (fecha de ingreso)
     */
    public function generaNoEmpleadoDAO($ano, $mes, $dia)
    {

        $fecha_ingreso = '%' . $ano . '-' . $mes . '-' . $dia . '%';


        $sql = "SELECT SQL_CALC_FOUND_ROWS * FROM usuarios WHERE Fecha_Ingreso LIKE :fecha_ingreso ORDER BY No_Empleado DESC LIMIT 1";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindParam(":fecha_ingreso", $fecha_ingreso, PDO::PARAM_STR);
        try {
            $stmt->execute();
            $response['status'] = true;
            $data = $stmt->fetch(PDO::FETCH_OBJ);
            $total = $this->_dbh->query('SELECT FOUND_ROWS();')->fetch(PDO::FETCH_COLUMN);
            // Generamos el no_empleado aleatoreo
            if ($total == 0) {
                $response['no_empleado'] = substr($ano, 2, 2) . $mes . str_pad($dia, 2, '0', STR_PAD_LEFT) . '001';
            } else {
                $siguiente = str_pad(substr($data->No_Empleado, -3) + 1, 3, '0', STR_PAD_LEFT);
                $response['no_empleado'] = substr($data->No_Empleado, 0, 6) . $siguiente;
            }

        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener fechas de ingreso de usuarios';
        }
        return $response;
    }


    /*
     * Método: validaUsuarioDAO
     * Descripción:  obtiene nuevo número de empleado
     * Param:  ano, mes, dia (fecha de ingreso)
     */
    public function validaUsuarioDAO($usuario, $idUsuario)
    {
        $sql = "SELECT SQL_CALC_FOUND_ROWS * FROM usuarios where Usuario = ? AND Id_Usuario != ?";

        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $usuario, PDO::PARAM_STR);
        $stmt->bindValue(2, $idUsuario, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $data = $stmt->fetch(PDO::FETCH_OBJ);
            $total = $this->_dbh->query('SELECT FOUND_ROWS();')->fetch(PDO::FETCH_COLUMN);
            if ($total == 0) {
                $response['existe'] = false;
            } else {
                $response['existe'] = true;
            }

        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los usuarios en validaUsuarioDAO';
        }

        return $response;
    }


    /*
     * Metodo regresaUsuarioDAO
     * @param idUsuario
     */

    public function regresaUsuarioDAO($idUsuario)
    {
        $sql = "SELECT * FROM `usuarios` WHERE `id_Usuario` = ? AND `Status` = 1";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idUsuario, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuario'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Ocurrió un error al obtener el usuario . ' . TICKETS;
        }

        return $response;
    }

    /*
    * Metodo altaUsuarioDAO
    * @param Usuario $u, Int $tipo
    * @return mixed array con los datos de respuesta
    * Descripción:
    *
    */
    public function altaUsuarioDAO(Usuario $usuario, $tipo)
    {
        if ($tipo == 1) {
            $sql = "INSERT INTO usuarios (Id_Usuario, Nombre, Apellido_Paterno, Apellido_Materno, Usuario, Password, 
            Tipo, Id_Direccion, Mail, Id_Sucursal, Status, Fecha_Ingreso , RFC , CURP, NSS, Id_Regimen_Contratacion, 
            Id_Banco_Pago_Nomina, Id_Riesgo_Puesto, Tipo_Contratacion, Departamento, Tipo_Jornada, Periodicidad_Pago, No_Empleado, Clabe, Id_Gerente, Mando, Id_Sesion, Codigo_Barras, Marketing, fechaPassword, passwordActualizada, Telefono, Licencia, Vigencia_Licencia) 
            VALUES (NULL,:nombre,:paterno,:materno,:usuario,:password,:perfil,:idDireccion,:mail,:idSucursal,1,:fechaIngreso,:rfc,:curp,:nss,
                    :regimenContratacion,:bancoPagoNomina,:riesgoPuesto,:tipoContratacion,:departamento,:tipoJornada,:periodicidadPago,:noEmpleado,:clabe, :gerente, :mando, :sesion, :codigo, :marketing, NOW(),0, 0, :licencia, :vigenciaLicencia)";
        } else {
            $sql = "UPDATE usuarios SET Nombre=:nombre, Apellido_Paterno=:paterno, Apellido_Materno=:materno, Usuario=:usuario, 
                                        Tipo=:perfil, Status=:status, RFC = :rfc, CURP=:curp, NSS=:nss, Mail=:mail, Id_Sucursal=:idSucursal, 
                                        Id_Regimen_Contratacion=:regimenContratacion, Id_Banco_Pago_Nomina=:bancoPagoNomina, Id_Riesgo_Puesto=:riesgoPuesto, 
                                        Tipo_Contratacion = :tipoContratacion, Departamento=:departamento, Tipo_Jornada=:tipoJornada, 
                                        Periodicidad_Pago = :periodicidadPago, Clabe = :clabe, fecha_baja = :fechaBaja, razon_baja = :razonBaja, Licencia = :licencia , Vigencia_Licencia = :vigenciaLicencia, regimen_fiscal = :regimen  WHERE Id_Usuario=:idUsuario";
        }

        $stmt = $this->_dbh->prepare($sql);

        $nombre = $usuario->__get('_nombre');
        $paterno = $usuario->__get('_paterno');
        $materno = $usuario->__get('_materno');
        $usuario_ = $usuario->__get('_usuario');
        $perfil = $usuario->__get('_tipo');
        $mail = $usuario->__get('_email');
        $idSucursal = $usuario->__get('_idSucursal');
        $rfc = $usuario->__get('_rfc');
        $curp = $usuario->__get('_curp');
        $nss = $usuario->__get('_nss');
        $regimenContratacion = $usuario->__get('_idRegimenContratacion');
        $bancoPagoNomina = $usuario->__get('_idBancoPagoNomina');
        $riesgoPuesto = $usuario->__get('_idRiesgoPuesto');
        $tipoContratacion = $usuario->__get('_tipoContratacion');
        $departamento = $usuario->__get('_departamento');
        $tipoJornada = $usuario->__get('_tipoJornada');
        $periodicidadPago = $usuario->__get('_periodicidadPago');
        $clabe = $usuario->__get('_clabe');
        $licencia = $usuario->__get('_licencia');
        $vigenciaLicencia = $usuario->__get('_vigenciaLicencia');
        $regimen = $usuario->__get('_regimen');

        /*
         * obtenemos valores y asignamos usuarios
         */
        $stmt->bindParam(":nombre", $nombre, PDO::PARAM_STR);
        $stmt->bindParam(":paterno", $paterno, PDO::PARAM_STR);
        $stmt->bindParam(":materno", $materno, PDO::PARAM_STR);
        $stmt->bindParam(":usuario", $usuario_, PDO::PARAM_STR);
        $stmt->bindParam(":perfil", $perfil, PDO::PARAM_INT);
        $stmt->bindParam(":mail", $mail, PDO::PARAM_STR);
        $stmt->bindParam(":idSucursal", $idSucursal, PDO::PARAM_INT);
        $stmt->bindParam(":rfc", $rfc, PDO::PARAM_STR);
        $stmt->bindParam(":curp", $curp, PDO::PARAM_STR);
        $stmt->bindParam(":nss", $nss, PDO::PARAM_STR);
        $stmt->bindParam(":regimenContratacion", $regimenContratacion, PDO::PARAM_INT);
        $stmt->bindParam(":bancoPagoNomina", $bancoPagoNomina, PDO::PARAM_INT);
        $stmt->bindParam(":riesgoPuesto", $riesgoPuesto, PDO::PARAM_INT);
        $stmt->bindParam(":tipoContratacion", $tipoContratacion, PDO::PARAM_STR);
        $stmt->bindParam(":departamento", $departamento, PDO::PARAM_STR);
        $stmt->bindParam(":tipoJornada", $tipoJornada, PDO::PARAM_STR);
        $stmt->bindParam(":periodicidadPago", $periodicidadPago, PDO::PARAM_STR);
        $stmt->bindParam(":clabe", $clabe, PDO::PARAM_STR);
        $stmt->bindParam(":licencia", $licencia, PDO::PARAM_STR);
        $stmt->bindParam(":vigenciaLicencia", $vigenciaLicencia, PDO::PARAM_STR);
        $stmt->bindParam(":regimen", $regimen, PDO::PARAM_STR);
        /*
         *
         */
        if ($tipo == 1) {
            $password = $usuario->__get('_contrasena');
            $fechaIngreso = $usuario->__get('_fechaRegistro');
            $noEmpleado = $usuario->__get('_noEmpleado');
            $idDireccion = $usuario->__get('_idDireccion');
            $gerente = $usuario->__get('_idGerente');
            $mando = $usuario->__get('_mando');
            $sesion = $usuario->__get('_idSesion');
            $marketing = $usuario->__get('_marketing');
            $codigo = $usuario->__get('_codigoBarras');
            /*
             *
             */
            $stmt->bindParam(":password", $password, PDO::PARAM_STR);
            $stmt->bindParam(":fechaIngreso", $fechaIngreso, PDO::PARAM_STR);
            $stmt->bindParam(":noEmpleado", $noEmpleado, PDO::PARAM_STR);
            $stmt->bindParam(":idDireccion", $idDireccion, PDO::PARAM_INT);
            $stmt->bindParam(":gerente", $gerente, PDO::PARAM_INT);
            $stmt->bindParam(":mando", $mando, PDO::PARAM_INT);
            $stmt->bindParam(":sesion", $sesion, PDO::PARAM_STR);
            $stmt->bindParam(":marketing", $marketing, PDO::PARAM_INT);
            $stmt->bindParam(":codigo", $codigo, PDO::PARAM_STR);
        }
        if ($tipo == 2) {

            $idUsuario = $usuario->__get('_idUsuario');
            $status = $usuario->__get('_status');
            $fechaBaja = $usuario->__get('_fechaBaja');
            $razonBaja = $usuario->__get('_razonBaja');

            $stmt->bindParam(":idUsuario", $idUsuario, PDO::PARAM_INT);
            $stmt->bindParam(":status", $status, PDO::PARAM_INT);
            $stmt->bindParam(":fechaBaja", $fechaBaja, PDO::PARAM_STR);
            $stmt->bindParam(":razonBaja", $razonBaja, PDO::PARAM_STR);
        }
        try {
            $stmt->execute();
            $response['status'] = true;
            if ($tipo == 1) {
                $response['id_usuario'] = $this->_dbh->lastInsertId();
            } else {
                $response['id_usuario'] = $usuario->__get('_idUsuario');
            }

        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al guardar los datos del usuario. ';
        }

        return $response;
    } // fin altaUsuarioDAO


    /*
   * Metodo altaNominaUsuarioDAO
   * Param: Obj $usuario y tipo 1 = alta, 2 = update
   * Desarrollo: AGRA
   * Descripción: regresa el id nomina o actualiza nomina por usuario
   */
    public function altaNominaUsuarioDAO(Nomina $nomina, $tipo)
    {
        if ($tipo == 1) {
            $sql = "INSERT INTO nomina (Id_Nomina, Id_Usuario, Sueldo, Dias_Aguinaldo, Dias_Vacaciones, Prima_Vacacional,
                                Pension_Alimenticia, Prestamo_Infonavit, Asistencia, Puntualidad, Despensa, forma_pago, servicio_region_fronteriza, entidad_de_servicio, tipo_prestamo_infonavit, tipo_nomina,
                                porcentaje_interna, porcentaje_externa, isr, Bandera_Puntualidad, Bandera_Asistencia, Bandera_Despensa)
                        VALUES (NULL, :idUsuario, :sueldo, :diasAguinaldo, :diasVacaciones, :primaVacacional,
                                :pensionAlimenticia, :prestamoInfonavit, :asistencia, :puntualidad, :despensa, :formapago, :servicioregionfronterizo, :entidad_de_servicio, 
                                :tipo_prestamo_infonavit,  :tipo_nomina,  :porcentaje_interno, :porcentaje_externo, :isr, :banderaPuntualidad, :banderaAsistencia, :banderaDespensa  )";
        } else {
            $sql = "UPDATE nomina SET Sueldo=:sueldo, Dias_Aguinaldo = :diasAguinaldo, Dias_Vacaciones=:diasVacaciones, 
                                      Prima_Vacacional=:primaVacacional, Pension_Alimenticia=:pensionAlimenticia, prestamo_infonavit=:prestamoInfonavit, 
                                      Asistencia = :asistencia, Puntualidad =:puntualidad, Despensa =:despensa, forma_pago = :formapago, servicio_region_fronteriza = :servicioregionfronterizo, entidad_de_servicio = :entidad_de_servicio, 
                                      tipo_prestamo_infonavit = :tipo_prestamo_infonavit, tipo_nomina = :tipo_nomina, porcentaje_interna = :porcentaje_interno, 
                                      porcentaje_externa = :porcentaje_externo, isr = :isr, Bandera_Puntualidad = :banderaPuntualidad, Bandera_Asistencia = :banderaAsistencia,
                                      Bandera_Despensa = :banderaDespensa  
                                      WHERE Id_Usuario =:idUsuario AND Id_Nomina= :idNomina";
        }
        $stmt = $this->_dbh->prepare($sql);

        /*
         * obtenemos los valores y asignamos variables
         */
        $idUsuario = $nomina->__get('_idUsuario');
        $sueldo = $nomina->__get('_sueldo');
        $diasAguinaldo = $nomina->__get('_diasAguinaldo');
        $diasVacaciones = $nomina->__get('_diasVacaciones');
        $primaVacacional = $nomina->__get('_primaVacacional');
        $pensionAlimenticia = $nomina->__get('_pensionAlimenticia');
        $prestamoInfonavit = $nomina->__get('_prestamoInfonavit');
        $asistencia = $nomina->__get('_asistencia');
        $puntualidad = $nomina->__get('_puntualidad');
        $despensa = $nomina->__get('_despensa');
        $formaPago = $nomina->__get('_forma_de_pago');
        $servicioFronterizo = $nomina->__get('_servicio_fronterizo');
        $entidad_de_servicio = $nomina->__get('_entidad_de_servicio');

        $tipo_prestamo_infonavit = $nomina->__get('_tipo_prestamo_infonavit');
        $tipo_nomina = $nomina->__get('_tipo_nomina');
//        $proveedor = $nomina->__get('_proveedor');
        $porcentaje_interno = $nomina->__get('_porcentaje_interno');
        $porcentaje_externo = $nomina->__get('_porcentaje_externo');
        $isr = $nomina->__get('_isr');
        $banderaAsistencia = $nomina->__get('_banderaAsistencia');
        $banderaPuntualidad = $nomina->__get('_banderaPuntualidad');
        $banderaDespensa = $nomina->__get('_banderaDespensa');
//        $comisiones = $nomina->__get('_comisiones');
//        $imms_patronal = $nomina->__get('_imms_patronal');



        $stmt->bindParam(":idUsuario", $idUsuario, PDO::PARAM_INT);
        $stmt->bindParam(":sueldo", $sueldo, PDO::PARAM_STR);
        $stmt->bindParam(":diasAguinaldo", $diasAguinaldo, PDO::PARAM_INT);
        $stmt->bindParam(":diasVacaciones", $diasVacaciones, PDO::PARAM_INT);
        $stmt->bindParam(":primaVacacional", $primaVacacional, PDO::PARAM_STR);
        $stmt->bindParam(":pensionAlimenticia", $pensionAlimenticia, PDO::PARAM_STR);
        $stmt->bindParam(":prestamoInfonavit", $prestamoInfonavit, PDO::PARAM_STR);
        $stmt->bindParam(":asistencia", $asistencia, PDO::PARAM_STR);
        $stmt->bindParam(":puntualidad", $puntualidad, PDO::PARAM_STR);
        $stmt->bindParam(":despensa", $despensa, PDO::PARAM_STR);
        $stmt->bindParam(":formapago", $formaPago, PDO::PARAM_STR);
        $stmt->bindParam(":servicioregionfronterizo", $servicioFronterizo, PDO::PARAM_STR);
        $stmt->bindParam(":entidad_de_servicio", $entidad_de_servicio, PDO::PARAM_STR);
        $stmt->bindParam(":tipo_prestamo_infonavit", $tipo_prestamo_infonavit, PDO::PARAM_STR);
        $stmt->bindParam(":tipo_nomina", $tipo_nomina, PDO::PARAM_STR);
//        $stmt->bindParam(":proveedor", $proveedor, PDO::PARAM_INT);
        $stmt->bindParam(":porcentaje_interno", $porcentaje_interno, PDO::PARAM_STR);
        $stmt->bindParam(":porcentaje_externo", $porcentaje_externo, PDO::PARAM_STR);
        $stmt->bindParam(":isr", $isr, PDO::PARAM_STR);
        $stmt->bindParam(":banderaAsistencia", $banderaAsistencia, PDO::PARAM_STR);
        $stmt->bindParam(":banderaPuntualidad", $banderaPuntualidad, PDO::PARAM_STR);
        $stmt->bindParam(":banderaDespensa", $banderaDespensa, PDO::PARAM_STR);
//        $stmt->bindParam(":comisiones", $comisiones, PDO::PARAM_INT);
//        $stmt->bindParam(":imms_patronal", $imms_patronal, PDO::PARAM_INT);
        if ($tipo == 2) {
            $idNomina = $nomina->__get('_idNomina');
            $stmt->bindParam(":idNomina", $idNomina, PDO::PARAM_INT);
        }

        try {
            $stmt->execute();
            $response['status'] = true;
            if ($tipo == 1) {
                $response['id_nomina'] = $this->_dbh->lastInsertId();
            } else {
                $response['id_nomina'] = $nomina->__get('_idNomina');
            }

        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al guardar la nomina del nuevo usuario. ';
        }

        return $response;
    } // fin altaNominaUsuarioDAO


    public function obtenerUsuarioNomina($idUsuario)
    {
        $sql = "SELECT * from nomina where Id_Usuario = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idUsuario, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['nomina'] = $stmt->fetch(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener  '
                . ' la nomina. ' . TICKETS;
        }

        return $response;
    }

    /** by isantosp **/
    public function getBancoNomina()
    {
        $sql = "SELECT * FROM banco_pago_nomina ORDER BY Clave";
        $stmt = $this->_dbh->prepare($sql);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['bancos'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos de los bancos';
        }

        return $response;
    }

    public function getRiesgoPuesto()
    {
        $sql = "SELECT * FROM riesgo_puesto ORDER BY Clave";
        $stmt = $this->_dbh->prepare($sql);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['riesgos'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos de los riesgos';
        }

        return $response;
    }

    public function getRegimenContratacion()
    {
        $sql = "SELECT * FROM regimen_contratacion";
        $stmt = $this->_dbh->prepare($sql);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['regimenes'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos de los regimenes';
        }

        return $response;
    }

    public function guardaPermisoDAO($user, $suc, $data)
    {
        $tipo = filter_var($data->tipo, FILTER_SANITIZE_STRING);
        if ($tipo == 'otro') $tipo = filter_var($data->otro, FILTER_SANITIZE_STRING);
        $desde = filter_var($data->desde, FILTER_SANITIZE_STRING);
        $hasta = filter_var($data->hasta, FILTER_SANITIZE_STRING);
        $dias = filter_var($data->dias, FILTER_SANITIZE_NUMBER_INT);
        $motivo = filter_var($data->motivo, FILTER_SANITIZE_STRING);
        $horaInicio = filter_var($data->horaInicio, FILTER_SANITIZE_STRING);
        $horaFin = filter_var($data->horaFin, FILTER_SANITIZE_STRING);

        $sql = "INSERT INTO asistencia_permiso 
                    (Id_Asistencia_Permiso, Tipo, Fecha_Inicio, Fecha_Fin, Hora_Inicial, Hora_Final,Dias_Solicitados, Motivo, 
                    Status, Id_Usuario, Id_Gerente, Comentario_Director, Id_Director, Fecha_Solicitud, Id_Sucursal) 
                    VALUES (NULL,:tipo,:desde,:hasta,:horaInicio,:horaFin,:dias,:motivo,
                            0,:user,0,'',0,NOW(),:suc) ";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindParam(':tipo', $tipo, PDO::PARAM_STR);
        $stmt->bindParam(':desde', $desde, PDO::PARAM_STR);
        $stmt->bindParam(':hasta', $hasta, PDO::PARAM_STR);
        $stmt->bindParam(':horaInicio', $horaInicio, PDO::PARAM_STR);
        $stmt->bindParam(':horaFin', $horaFin, PDO::PARAM_STR);
        $stmt->bindParam(':dias', $dias, PDO::PARAM_INT);
        $stmt->bindParam(':motivo', $motivo, PDO::PARAM_STR);
        $stmt->bindParam(':user', $user, PDO::PARAM_INT);
        $stmt->bindParam(':suc', $suc, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['permiso'] = $this->_dbh->lastInsertId();
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al guardar el permiso. ' . TICKETS;
        }

        return $response;
    }

    public function buscaarVendedoresActivos($vendedor)
    {
        $sql = "SELECT * FROM `usuarios` WHERE (`Nombre` LIKE ? OR `Apellido_Paterno` LIKE ? OR `Apellido_Materno` LIKE ?) AND `Status` = 1 AND (`Tipo` = 2 OR `Tipo` = 4)";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, "%$vendedor%", PDO::PARAM_STR);
        $stmt->bindValue(2, "%$vendedor%", PDO::PARAM_STR);
        $stmt->bindValue(3, "%$vendedor%", PDO::PARAM_STR);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['vendedores'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al buscar el vendedor. ' . TICKETS;
        }

        return $response;
    }

    public function obtenerPermisosLaboralesDirector($idSucursal)
    {
        $sql = "SELECT `usuarios`.`Nombre`, `usuarios`.`Apellido_Paterno`, `usuarios`.`Apellido_Materno`, `asistencia_permiso`.`Tipo`, `asistencia_permiso`.`Fecha_Solicitud`, `asistencia_permiso`.`Fecha_Inicio`, `asistencia_permiso`.`Fecha_Fin`, `asistencia_permiso`.`Hora_Inicial`, `asistencia_permiso`.`Hora_Final`, `asistencia_permiso`.`Dias_Solicitados`, `asistencia_permiso`.`Motivo`, `asistencia_permiso`.`Id_Asistencia_Permiso` FROM `asistencia_permiso` INNER JOIN `usuarios` ON `usuarios`.`Id_Usuario` = `asistencia_permiso`.`Id_Usuario` WHERE `asistencia_permiso`.`Status` = 0 ";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idSucursal, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['permisos'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener las solicitudes. ' . TICKETS;
        }

        return $response;
    }

    public function obtenerPermisosLaboralesGerente($idSucursal, $idGerente)
    {
        $sql = "SELECT `usuarios`.`Nombre`, `usuarios`.`Apellido_Paterno`, `usuarios`.`Apellido_Materno`, `asistencia_permiso`.`Tipo`, `asistencia_permiso`.`Fecha_Solicitud`, `asistencia_permiso`.`Fecha_Inicio`, `asistencia_permiso`.`Fecha_Fin`, `asistencia_permiso`.`Dias_Solicitados`, `asistencia_permiso`.`Motivo`, `asistencia_permiso`.`Id_Asistencia_Permiso` FROM `asistencia_permiso` INNER JOIN `usuarios` ON `usuarios`.`Id_Usuario` = `asistencia_permiso`.`Id_Usuario` WHERE `usuarios`.`Id_Gerente` = ? AND `asistencia_permiso`.`Id_Sucursal` = ? AND `asistencia_permiso`.`Status` = 0";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idGerente, PDO::PARAM_INT);
        $stmt->bindValue(2, $idSucursal, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['permisos'] = $stmt->fetchAll();
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener las solicitudes. ' . TICKETS;
        }

        return $response;
    }

    public function cambiaStatusPermiso($status, $idPermiso)
    {
        $sql = "UPDATE `asistencia_permiso` SET `Status` = ? WHERE `Id_Asistencia_Permiso` = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $status, PDO::PARAM_INT);
        $stmt->bindValue(2, $idPermiso, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al cambiar el status de la solicitud. ' . TICKETS;
        }

        return $response;
    }

    public function actualizaAutorizacionDirector($idUsuario, $idPermiso)
    {
        $sql = "UPDATE `asistencia_permiso` SET `Id_Director` = ? WHERE `Id_Asistencia_Permiso` = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idUsuario, PDO::PARAM_INT);
        $stmt->bindValue(2, $idPermiso, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al cambiar el status de la solicitud. ' . TICKETS;
        }

        return $response;
    }

    public function actualizaAutorizacionGerente($idUsuario, $idPermiso)
    {
        $sql = "UPDATE `asistencia_permiso` SET `Id_Gerente` = ? WHERE `Id_Asistencia_Permiso` = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idUsuario, PDO::PARAM_INT);
        $stmt->bindValue(2, $idPermiso, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al cambiar el status de la solicitud. ' . TICKETS;
        }

        return $response;
    }
    
    public function obtenerPermisosAsistenciaPendientesAutorizacion()
    {
        $sql = "SELECT * FROM `asistencia_permiso` WHERE `Status` = 0";
        $stmt = $this->_dbh->prepare($sql);
        try {
            $stmt->execute();
            $response['permisos'] = $stmt->fetchAll(PDO::FETCH_OBJ);
            $response['total'] = $stmt->rowCount();
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos de la solicitud. ' . TICKETS;
        }

        return $response;
    }    

    public function obtenerPermisoLaboralPorId($idPermiso)
    {
        $sql = "SELECT * FROM `asistencia_permiso` WHERE `Id_Asistencia_Permiso` = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idPermiso, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['permiso'] = $stmt->fetch(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos de la solicitud. ' . TICKETS;
        }

        return $response;
    }

    public function obtenerUsuariosBloqueados()
    {
        $sql = "SELECT * FROM bloqueos_usuarios as bu "
            . " inner join usuarios as u on u.Id_Usuario = bu.id_usuario "
            . " WHERE bu.desbloqueo = 0 ORDER BY u.Usuario ASC";
        $stmt = $this->_dbh->prepare($sql);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos de la solicitud. ' . TICKETS;
        }

        return $response;
    }

    public function obtenerUsuariosActivosSucursal($idSucursal)
    {
        $sql = "SELECT * FROM usuarios WHERE Status = 1 AND Id_Sucursal = ? AND (Id_Usuario <> 16 AND Id_Usuario <> 17 and Id_Usuario != 28 and Id_Usuario != 27 and Id_Usuario != 29 and Id_Usuario != 122)";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idSucursal, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            $response['status'] = false;
            $response['error'] = 'Ocurrió un error al obtener los usuarios de la sucursal. ';
        }

        return $response;
    }

    public function obtenerPermisoLaboralUsuario($idUsuario, $fecha)
    {
        $sql = "SELECT * FROM asistencia_permiso WHERE Id_Usuario = ? AND (? between Fecha_Inicio AND Fecha_Fin) AND Status = 1";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idUsuario, PDO::PARAM_INT);
        $stmt->bindValue(2, $fecha, PDO::PARAM_STR);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['hay'] = $stmt->rowCount();
            $response['permiso'] = $stmt->fetch(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            $response['status'] = false;
            $response['error'] = 'Ocurrió un error al validar el permiso';
        }

        return $response;
    }

    public function obtenerUsuariosPorTipoYSucursal($tipo, $idSucursal)
    {
        $sql = "select * from usuarios where Status > 0 and Tipo = ? and Id_Sucursal = ? and Id_Usuario <> 111 limit 0, 1";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $tipo, PDO::PARAM_INT);
        $stmt->bindValue(2, $idSucursal, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetch(PDO::FETCH_OBJ);
            $response['total'] = $stmt->rowCount();
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos del usuario por tipo y sucursal';
        }

        return $response;
    }

    public function obtenerUsuariosNoInactivos()
    {
        $sql = "select * from usuarios where Status > 0 and Id_Usuario != 28 and Id_Usuario != 27 and Id_Usuario != 29 and Id_Usuario != 122 ";
        $stmt = $this->_dbh->prepare($sql);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos del usuario';
        }

        return $response;
    }

    public function obtenerResponsableParaPopsa($idSucursal, $tipo)
    {
        $sql = "SELECT * FROM `usuarios` WHERE `Status` >= 1 AND `Id_Sucursal` = ? AND `Tipo` = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idSucursal, PDO::PARAM_INT);
        $stmt->bindValue(2, $tipo, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los responsables';
        }

        return $response;
    }


    public function obtenerUsuarioPorId($idUsuario)
    {
        $sql = "SELECT * FROM `usuarios` WHERE `Id_Usuario` = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idUsuario, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, no pude obtener los datos del usuario. '
                . TICKETS;
        }

        return $response;
    }

    public function obtenerVendedoresActivosPorSucursalPrueba($idSucursal)
    {
        $sql = "SELECT * FROM `usuarios` WHERE (`Tipo` = 4 OR `Tipo` = 2 or Id_Usuario = 16)  AND `Status` = 1 AND `Id_Sucursal` = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idSucursal, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['vendedores'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener la lista '
                . 'de vendedores. ' . TICKETS;
        }

        return $response;
    }

    public function obtenerDiaFestivo($fecha)
    {
        $sql = "SELECT * FROM dias_festivos WHERE dia_festivo = ? ";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $fecha, PDO::PARAM_STR);

        try {
            $stmt->execute();
            $response['status'] = true;
            $response['hay'] = $stmt->rowCount();
        } catch (PDOException $e) {
            $response['status'] = false;
            $response['error'] = 'Ocurrió un error al obtener la fecha del dia festivo';
        }

        return $response;
    }

    public function agregarBloqueoUsuario($idUsuario, $tipoBloqueo)
    {
        if(( $idUsuario != 132 and $idUsuario != 63) || ($tipoBloqueo == "Secuencia Semanal no enviada autorizacion" || $tipoBloqueo == "Secuencia Semanal" || $tipoBloqueo == "Forecast" ||
                $tipoBloqueo == "No completo actividad de secuencia" || $tipoBloqueo == "cierre popsa" || $tipoBloqueo == "Forecast Anual" || $tipoBloqueo == "Autorizacion Forecast Anual" || $tipoBloqueo == "curso no programado")) {
            if($idUsuario != 16 and $idUsuario != 17 and $idUsuario != 29 and $idUsuario != 28  and  $idUsuario != 27) {
                if ($idUsuario != 130 and  $idUsuario != 136 and $idUsuario != 103) {
                    $sql = "INSERT INTO `bloqueos_usuarios` (`id_usuario`, `desbloqueo`, `tipo_bloqueo`)"
                        . " VALUES (?, 0, ?)";

                    $stmt = $this->_dbh->prepare($sql);
                    $stmt->bindValue(1, $idUsuario, PDO::PARAM_INT);
                    $stmt->bindValue(2, $tipoBloqueo, PDO::PARAM_STR);

                    try {
                        $stmt->execute();
                        $response['status'] = true;
                    } catch (PDOException $e) {
                        Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
                        $response['status'] = false;
                        $response['error'] = 'Lo siento, ocurrió un error al bloquear al usuario ' . TICKETS;
                    }

                    return $response;
                }
            }
        }
    }

    public function desbloqueaUsuario($id_bloqueos_usuarios)
    {
        $sql = "UPDATE `bloqueos_usuarios` SET `desbloqueo` = 1 WHERE `id_usuario` = ? and `desbloqueo` = 0";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $id_bloqueos_usuarios, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Ocurrió un error al desbloquear al usuario. ' . TICKETS;
        }

        return $response;
    }

    public function esUsuarioSinBloqueos(Usuario $u)
    {
        $sql = "SELECT * from bloqueos_usuarios as bu "
            . "  inner join usuarios as u on u.Id_Usuario = bu.id_usuario "
            . " where u.Usuario = ? and bu.desbloqueo = 0";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $u->__get('_usuario'), PDO::PARAM_STR);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['total'] = $stmt->rowCount();
            $response['datos'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), __FILE__, __LINE__);
            $response['status'] = false;
            $response['error'] = 'Lo siento, no pude validar los datos de usuario. '
                . TICKETS;
        }

        return $response;
    }

    public function VerificarBloqueoUsuario($idUsuario, $tipoBloqueo)
    {
        $sql = "select * from bloqueos_usuarios as bu 
inner join usuarios as u on u.Id_Usuario = bu.id_usuario
where bu.id_usuario = ? and ((bu.tipo_bloqueo = ? and bu.desbloqueo = 0) || u.Status != 1);";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idUsuario, PDO::PARAM_INT);
        $stmt->bindValue(2, $tipoBloqueo, PDO::PARAM_STR);

        try {
            $stmt->execute();
            $response['status'] = true;
            $response['total'] = $stmt->rowCount();
            $response['datos'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), __FILE__, __LINE__);
            $response['status'] = false;
            $response['error'] = 'Lo siento, no pude validar los datos del bloqueo del usuario. '
                . TICKETS;
        }

        return $response;
    }

    public function obtenermotivosUsuarioBloqueado($idUsuario)
    {
        $sql = " select * from bloqueos_usuarios  "
            . " where id_usuario = ? and desbloqueo = 0 ";
        $stmt = $this->_dbh->prepare($sql);

        $stmt->bindValue(1, $idUsuario, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos de la solicitud. ' . TICKETS;
        }

        return $response;
    }

    public function obtenerIdUsuariosBloqueados()
    {
        $sql = "SELECT u.Id_Usuario, u.Nombre, u.Apellido_Paterno,u.Apellido_Materno FROM bloqueos_usuarios as bu "
            . " inner join usuarios as u on u.Id_Usuario = bu.id_usuario "
            . " WHERE bu.desbloqueo = 0 GROUP BY u.Id_Usuario ORDER BY u.Usuario ASC";
        $stmt = $this->_dbh->prepare($sql);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos de la solicitud. ' . TICKETS;
        }

        return $response;
    }

    public function totalUsuariosActivos()
    {
        $sql = "SELECT COUNT(`Id_Usuario`) FROM `usuarios` WHERE `Status` = 1";
        $stmt = $this->_dbh->prepare($sql);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['total'] = $stmt->fetchColumn();
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento no pude validar el total de usuarios de la '
                . 'sucursal. ' . TICKETS;
        }

        return $response;
    }

    public function obtenerUsuariosActivosListado($inicio, $registros)
    {
        $sql = "SELECT * FROM `usuarios` WHERE `Status` = 1 ORDER BY Status DESC LIMIT ?, ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $inicio, PDO::PARAM_INT);
        $stmt->bindValue(2, $registros, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos de usuarios';
        }

        return $response;
    }

    public function totalCompletarUsuarioActivo($usuario, $inicio, $registros)
    {

        $sql = "SELECT SQL_CALC_FOUND_ROWS * FROM `usuarios` WHERE (Usuario LIKE ? 
                        OR CONCAT(Nombre,' ',Apellido_Paterno,' ',Apellido_Materno) LIKE ? 
                        OR Mail LIKE ? ) and `Status` = 1
                        LIMIT ?, ?";
        $stmt = $this->_dbh->prepare($sql);
        //$stmt->bindValue(1, $usuario, PDO::PARAM_STR );
        $stmt->bindValue(1, "%$usuario%", PDO::PARAM_STR);
        $stmt->bindValue(2, "%$usuario%", PDO::PARAM_STR);
        $stmt->bindValue(3, "%$usuario%", PDO::PARAM_STR);
        $stmt->bindValue(4, $inicio, PDO::PARAM_INT);
        $stmt->bindValue(5, $registros, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
            $response['total'] = $this->_dbh->query('SELECT FOUND_ROWS();')->fetch(PDO::FETCH_COLUMN);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al buscar los usuarios';
        }

        return $response;

    }

    public function actualizarNominaUsuario(Nomina $nomina)
    {

        $sql = "UPDATE nomina SET Sueldo=:sueldo, forma_pago = :formapago, servicio_region_fronteriza = :servicioregionfronterizo, entidad_de_servicio = :entidad_de_servicio, 
                                Dias_Aguinaldo = :dias_aguinaldo, Dias_Vacaciones = :dias_vacaciones, Prima_Vacacional = :prima_vacacional, sueldo_integrado = :sueldo_integrado
                                WHERE Id_Usuario =:idUsuario ";

        $stmt = $this->_dbh->prepare($sql);

        $idUsuario = $nomina->__get('_idUsuario');
        $sueldo = $nomina->__get('_sueldo');
        $formaPago = $nomina->__get('_forma_de_pago');
        $servicioFronterizo = $nomina->__get('_servicio_fronterizo');
        $entidad_de_servicio = $nomina->__get('_entidad_de_servicio');

        $dias_aguinaldo = $nomina->__get('_diasAguinaldo');
        $dias_vacaciones = $nomina->__get('_diasVacaciones');
        $prima_vacacional = $nomina->__get('_primaVacacional');
        $sueldo_integrado = $nomina->__get('_sueldo_integrado');



        $stmt->bindParam(":idUsuario", $idUsuario, PDO::PARAM_INT);
        $stmt->bindParam(":sueldo", $sueldo, PDO::PARAM_STR);
        $stmt->bindParam(":formapago", $formaPago, PDO::PARAM_STR);
        $stmt->bindParam(":servicioregionfronterizo", $servicioFronterizo, PDO::PARAM_STR);
        $stmt->bindParam(":entidad_de_servicio", $entidad_de_servicio, PDO::PARAM_STR);
        $stmt->bindParam(":dias_aguinaldo", $dias_aguinaldo, PDO::PARAM_INT);
        $stmt->bindParam(":dias_vacaciones", $dias_vacaciones, PDO::PARAM_INT);
        $stmt->bindParam(":prima_vacacional", $prima_vacacional, PDO::PARAM_STR);
        $stmt->bindParam(":sueldo_integrado", $sueldo_integrado, PDO::PARAM_STR);

        try {
            $stmt->execute();
            $response['status'] = true;

        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al actualizar la nomina del usuario. ';
        }

        return $response;

    }

    public function actualizarInformacionNominaUsuario(Usuario $usuario)
    {

            $sql = "UPDATE usuarios SET  RFC = :rfc, CURP=:curp, NSS=:nss, 
                                        Id_Regimen_Contratacion=:regimenContratacion,
                                        Tipo_Contratacion = :tipoContratacion, Tipo_Jornada=:tipoJornada, 
                                        Periodicidad_Pago = :periodicidadPago, Fecha_Ingreso = :fechaIngreso WHERE Id_Usuario=:idUsuario";


        $stmt = $this->_dbh->prepare($sql);

        $rfc = $usuario->__get('_rfc');
        $curp = $usuario->__get('_curp');
        $nss = $usuario->__get('_nss');
        $regimenContratacion = $usuario->__get('_idRegimenContratacion');
        $tipoContratacion = $usuario->__get('_tipoContratacion');
        $tipoJornada = $usuario->__get('_tipoJornada');
        $periodicidadPago = $usuario->__get('_periodicidadPago');
        $fechaIngreso = $usuario->__get('_fechaRegistro');
        $idUsuario = $usuario->__get('_idUsuario');


        $stmt->bindParam(":rfc", $rfc, PDO::PARAM_STR);
        $stmt->bindParam(":curp", $curp, PDO::PARAM_STR);
        $stmt->bindParam(":nss", $nss, PDO::PARAM_STR);
        $stmt->bindParam(":tipoContratacion", $tipoContratacion, PDO::PARAM_STR);
        $stmt->bindParam(":tipoJornada", $tipoJornada, PDO::PARAM_STR);
        $stmt->bindParam(":periodicidadPago", $periodicidadPago, PDO::PARAM_STR);
        $stmt->bindParam(":regimenContratacion", $regimenContratacion, PDO::PARAM_INT);
        $stmt->bindParam(":fechaIngreso", $fechaIngreso, PDO::PARAM_STR);
        $stmt->bindParam(":idUsuario", $idUsuario, PDO::PARAM_INT);

        try {
            $stmt->execute();
            $response['status'] = true;

        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al actualizar los datos del usuario. ';
        }

        return $response;
    }

    public function obtenerTodosUsuariosActivosConNomina()
    {
        $sql = " SELECT * FROM usuarios 
                inner join nomina on nomina.Id_Usuario = usuarios.Id_Usuario
                WHERE usuarios.Status = 1 || usuarios.Status = -1";
        $stmt = $this->_dbh->prepare($sql);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos de usuarios';
        }

        return $response;
    }

    public function obtenerTodosUsuariosParaNominaMasiva($fechaInicial, $fechaFinal)
    {
        $sql = "SELECT u.*, n.* FROM usuarios as u
inner join nomina as n on n.Id_Usuario = u.Id_Usuario
 WHERE (u.Status = 1 or u.Status = -1)
         and not exists(select * from autorizacion_nomina as an where an.id_usuario = u.Id_Usuario and an.fecha_inicial >= ? and an.fecha_final <= ?)
          and (n.tipo_nomina = 'interna' || n.tipo_nomina = 'externa' || n.tipo_nomina = 'mixta')";

        $stmt = $this->_dbh->prepare($sql);

        $stmt->bindValue(1, $fechaInicial, PDO::PARAM_STR);
        $stmt->bindValue(2, $fechaFinal, PDO::PARAM_STR);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos de usuarios';
        }

        return $response;
    }

    public function esUsuarioSinBloqueosPorId($idUsuario)
    {
        $sql = "SELECT * from bloqueos_usuarios as bu "
            . "  inner join usuarios as u on u.Id_Usuario = bu.id_usuario "
            . " where u.Id_Usuario = ? and bu.desbloqueo = 0";
        $stmt = $this->_dbh->prepare($sql);

        $stmt->bindValue(1, $idUsuario, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['total'] = $stmt->rowCount();
            $response['datos'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), __FILE__, __LINE__);
            $response['status'] = false;
            $response['error'] = 'Lo siento, no pude validar los datos de usuario. '
                . TICKETS;
        }

        return $response;
    }

    public function obtenerFaltas($idUsuario, $fechainicial, $fechaFinal)
    {
        $sql = "select * from asistencia where Id_Usuario = ? and fecha between ? and ? and Asistencia = 3 ";
        $stmt = $this->_dbh->prepare($sql);

        $stmt->bindValue(1, $idUsuario, PDO::PARAM_INT);
        $stmt->bindValue(2, $fechainicial, PDO::PARAM_STR);
        $stmt->bindValue(3, $fechaFinal, PDO::PARAM_STR);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['total'] = $stmt->rowCount();
           // $response['datos'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), __FILE__, __LINE__);
            $response['status'] = false;
            $response['error'] = 'Lo siento, no pude obtener los dias injustificados. '
                . TICKETS;
        }

        return $response;
    }

    public function obtenerTarjetaAsistencia($idUsuario, $fecha_inicio, $fecha_fin)
    {
        $sql = "select * from asistencia where Id_Usuario = ? and fecha between ? and ? ";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idUsuario, PDO::PARAM_INT);
        $stmt->bindValue(2, $fecha_inicio, PDO::PARAM_STR);
        $stmt->bindValue(3, $fecha_fin, PDO::PARAM_STR);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['asistencia'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            $response['status'] = false;
            $response['error'] = 'Ocurrió un error al obtener la tarjeta de asistencia ';
        }

        return $response;
    }

    public function actualizarAsistencia($idAsistencia, $asistencia, $observasion)
    {
        $sql = "UPDATE `asistencia` SET `Asistencia` = ?, Observaciones = ? WHERE `Id_Asistencia` = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $asistencia, PDO::PARAM_INT);
        $stmt->bindValue(2, $observasion, PDO::PARAM_STR);
        $stmt->bindValue(3, $idAsistencia, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Ocurrió un error al actualizar la asistencia. ' . TICKETS;
        }

        return $response;
    }

    public function obtenerUsuariosNomina()
    {
        $sql = "SELECT * FROM `usuarios` WHERE `Status` = 1 || `Status` = -1 ";
        $stmt = $this->_dbh->prepare($sql);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos de usuarios';
        }

        return $response;
    }

    public function obtenerTotalDeVacacionesPorUsuarioYFechas($idUsuario, $fechaInicio, $fechaFin) {
        $sql = "select ap.*, concat(u.Nombre,' ',u.Apellido_Paterno, ' ', u.Apellido_Materno) as nombre_usuario from asistencia_permiso as ap
inner join usuarios as u on u.Id_Usuario = ap.Id_Usuario 
where ap.Tipo = 'vacaciones'
and ap.Status = 1 
and ap.Id_Usuario = :idUsuario ";
        if ($fechaInicio != "") {
            $sql .= "and ap.Fecha_Inicio >= :fechaInicial ";
        }
        if ($fechaFin != "") {
            $sql .= "and ap.Fecha_Fin <= :fechaFinal ";
        }
        $sql .= "order by ap.Fecha_Inicio;";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(':idUsuario', $idUsuario, PDO::PARAM_INT);
        if ($fechaInicio != "") {
            $stmt->bindValue(':fechaInicial', $fechaInicio, PDO::PARAM_STR);
        }
        if ($fechaFin != "") {
            $stmt->bindValue(':fechaFinal', $fechaFin, PDO::PARAM_STR);
        }
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['total'] = $stmt->rowCount();
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al contar los datos del historial de vacaciones. ' . TICKETS;
        }
        return $response;
    }

    public function obtenerHistorialDeVacacionesPorUsuarioYFechas($idUsuario, $fechaInicio, $fechaFin, $inicio, $registros) {
        $sql = "select ap.*, concat(u.Nombre,' ',u.Apellido_Paterno, ' ', u.Apellido_Materno) as nombre_usuario from asistencia_permiso as ap
inner join usuarios as u on u.Id_Usuario = ap.Id_Usuario 
where ap.Tipo = 'vacaciones'
and ap.Status = 1 
and ap.Id_Usuario = :idUsuario ";
        if ($fechaInicio != "") {
            $sql .= "and ap.Fecha_Inicio >= :fechaInicial ";
        }
        if ($fechaFin != "") {
            $sql .= "and ap.Fecha_Fin <= :fechaFinal ";
        }
        $sql .= "order by ap.Fecha_Inicio limit :inicio, :registros;";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(':idUsuario', $idUsuario, PDO::PARAM_INT);
        if ($fechaInicio != "") {
            $stmt->bindValue(':fechaInicial', $fechaInicio, PDO::PARAM_STR);
        }
        if ($fechaFin != "") {
            $stmt->bindValue(':fechaFinal', $fechaFin, PDO::PARAM_STR);
        }
        $stmt->bindValue(':inicio', $inicio, PDO::PARAM_INT);
        $stmt->bindValue(':registros', $registros, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['historial'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener el historial de vacaciones. ' . TICKETS;
        }
        return $response;
    }


    public function obtenerVacacionesSolicitadas($idUsuario, $fechaInicio, $fechaFinal )
    {
        $sql = "select * from asistencia_permiso where Id_Usuario = ? and Fecha_Inicio >= ? and Fecha_Inicio <= ? and Status = 1 and Tipo = 'vacaciones' ";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idUsuario, PDO::PARAM_INT);
        $stmt->bindValue(2, $fechaInicio, PDO::PARAM_STR);
        $stmt->bindValue(3, $fechaFinal, PDO::PARAM_STR);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['total'] = $stmt->rowCount();
            $response['vacaciones'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, no pude obtener los datos de las vacaciones. '
                . TICKETS;
        }

        return $response;
    }

    public function getPrivedoresInternosActivos()
    {
        $sql = "select * from proveedores_indirectos
where proveedores_indirectos.Status = 1
and proveedores_indirectos.nomina = 1 
order by proveedores_indirectos.Nombre;";
        $stmt = $this->_dbh->prepare($sql);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['proveedores'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos de los provedores internos';
        }

        return $response;
    }

    public function obtenerTotalDeNominasPorUsuarioYFechas($idUsuario, $fechaInicio, $fechaFin) {
        $sql = "select an.*, dan.total, dan.status_Pago, d.Folio_Documento from autorizacion_nomina as an
inner join detalle_autorizacion_nomina as dan on dan.id_autorizacion_nomina = an.id_autorizacion_nomina
left join documentos as d on d.Id_Documento = dan.id_documento 
where (dan.status_Pago = 'autorizado' or dan.status_Pago = 'pagado' or dan.status_Pago = 'timbrado')
and an.id_usuario = :idusuario ";
        if ($fechaInicio != "") {
            $sql .= "and an.fecha_inicial >= :fechaInicial ";
        }
        if ($fechaFin != "") {
            $sql .= "and an.fecha_final <= :fechaFinal ";
        }
        $sql .= "order by an.fecha_inicial;";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(':idusuario', $idUsuario, PDO::PARAM_INT);
        if ($fechaInicio != "") {
            $stmt->bindValue(':fechaInicial', $fechaInicio, PDO::PARAM_STR);
        }
        if ($fechaFin != "") {
            $stmt->bindValue(':fechaFinal', $fechaFin, PDO::PARAM_STR);
        }
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['total'] = $stmt->rowCount();
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al contar los datos del historial de nominas. ' . TICKETS;
        }
        return $response;
    }
    public function obtenerHistorialDeNominasPorUsuarioYFechas($idUsuario, $fechaInicio, $fechaFin, $inicio, $registros) {
        $sql = "select an.*, dan.total, dan.status_Pago, d.Folio_Documento, dan.tipo_nomina as tipoDeNomina from autorizacion_nomina as an
inner join detalle_autorizacion_nomina as dan on dan.id_autorizacion_nomina = an.id_autorizacion_nomina 
left join documentos as d on d.Id_Documento = dan.id_documento
where (dan.status_Pago = 'autorizado' or dan.status_Pago = 'pagado' or dan.status_Pago = 'timbrado')
and an.id_usuario = :idusuario ";
        if ($fechaInicio != "") {
            $sql .= "and an.fecha_inicial >= :fechaInicial ";
        }
        if ($fechaFin != "") {
            $sql .= "and an.fecha_final <= :fechaFinal ";
        }
        $sql .= "order by an.fecha_pago DESC limit :inicio, :registros;";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(':idusuario', $idUsuario, PDO::PARAM_INT);
        if ($fechaInicio != "") {
            $stmt->bindValue(':fechaInicial', $fechaInicio, PDO::PARAM_STR);
        }
        if ($fechaFin != "") {
            $stmt->bindValue(':fechaFinal', $fechaFin, PDO::PARAM_STR);
        }
        $stmt->bindValue(':inicio', $inicio, PDO::PARAM_INT);
        $stmt->bindValue(':registros', $registros, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['historial'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener el historial de nominas. ' . TICKETS;
        }
        return $response;
    }

    public function obtenerTotalDeTarjetasFirmadasPorFechasYUsuario($idUsuario, $fechaInicio, $fechaFin) {
        $sql = "select * from autorizacion_nomina
where status = 'aprobado'
and archivo_tarjeta_asistencia is not null
and id_usuario = :idUsuario ";
        if ($fechaInicio != "") {
            $sql .= "and fecha_inicio_corte >= :fechaInicial ";
        }
        if ($fechaFin != "") {
            $sql .= "and fecha_final_corte <= :fechaFinal ";
        }
        $sql .= "order by fecha_inicio_corte;";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(':idUsuario', $idUsuario, PDO::PARAM_INT);
        if ($fechaInicio != "") {
            $stmt->bindValue(':fechaInicial', $fechaInicio, PDO::PARAM_STR);
        }
        if ($fechaFin != "") {
            $stmt->bindValue(':fechaFinal', $fechaFin, PDO::PARAM_STR);
        }
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['total'] = $stmt->rowCount();
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al contar los datos del historial de tarjetas firmadas. ' . TICKETS;
        }
        return $response;
    }

    public function obtenerHistorialDeTarjetasFirmadasPorFechasYUsuario($idUsuario, $fechaInicio, $fechaFin, $inicio, $registros) {
        $sql = "select * from autorizacion_nomina
where status = 'aprobado'
and archivo_tarjeta_asistencia is not null
and id_usuario = :idUsuario ";
        if ($fechaInicio != "") {
            $sql .= "and fecha_inicio_corte >= :fechaInicial ";
        }
        if ($fechaFin != "") {
            $sql .= "and fecha_final_corte <= :fechaFinal ";
        }
        $sql .= "order by fecha_inicio_corte limit :inicio, :registros;";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(':idUsuario', $idUsuario, PDO::PARAM_INT);
        if ($fechaInicio != "") {
            $stmt->bindValue(':fechaInicial', $fechaInicio, PDO::PARAM_STR);
        }
        if ($fechaFin != "") {
            $stmt->bindValue(':fechaFinal', $fechaFin, PDO::PARAM_STR);
        }
        $stmt->bindValue(':inicio', $inicio, PDO::PARAM_INT);
        $stmt->bindValue(':registros', $registros, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['historial'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener el historial de tarjetas firmadas. ' . TICKETS;
        }
        return $response;
    }

    public function obtenerClaveRegimenContratacion($idRegimen)
    {
        $sql = "select * from regimen_contratacion where Id_Regimen_Contratacion = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1,$idRegimen, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['regimen'] = $stmt->fetch(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), __FILE__, __LINE__);
            $response['status'] = false;
            $response['error'] = 'Lo siento, no pude obtener el regimen. '
                . TICKETS;
        }

        return $response;
    }


    public function obtenerGerentesCronSaldos()
    {
        $sql = "select * from usuarios where Tipo = 2 and Status = 1 and Usuario !='amguijarro'";
        $stmt = $this->_dbh->prepare($sql);

        try {
            $stmt->execute();
            $response['status'] = true;
            $response['gerentes'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), __FILE__, __LINE__);
            $response['status'] = false;
            $response['error'] = 'Lo siento, no pude obtener el regimen. '
                . TICKETS;
        }

        return $response;
    }

    public function obtenerGerentePorSucursal($idSucursal)
    {
        $sql = "select * from usuarios where Tipo = 2 and Status = 1 and Id_Sucursal = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1,$idSucursal, PDO::PARAM_INT);


        try {
            $stmt->execute();
            $response['status'] = true;
            $response['gerente'] = $stmt->fetch(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), __FILE__, __LINE__);
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrio un error al obtener el gerente. '
                . TICKETS;
        }

        return $response;
    }

    public function obtenerGerentes()
    {
        $sql = "SELECT * FROM `usuarios` WHERE  `Tipo` = 2 AND `Status` >= 1";
        $stmt = $this->_dbh->prepare($sql);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['gerentes'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener la lista '
                . 'de gerentes. ' . TICKETS;
        }

        return $response;
    }

    public function obtenerIdUsuariosBloqueadosPorFecha($fechaInicio, $fechaFinal)
    {
        $sql = " SELECT u.Id_Usuario, u.Nombre, u.Apellido_Paterno,u.Apellido_Materno FROM bloqueos_usuarios as bu 
             inner join usuarios as u on u.Id_Usuario = bu.id_usuario 
             WHERE bu.desbloqueo = 0 and bu.created_at between ? and ? group by u.Id_Usuario ORDER BY u.Usuario ASC";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1,$fechaInicio, PDO::PARAM_STR);
        $stmt->bindValue(2,$fechaFinal, PDO::PARAM_STR);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['total'] = $stmt->rowCount();
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos de la solicitud. ' . TICKETS;
        }

        return $response;
    }

    public function obtenerUsuariosCursos()
    {
        $sql = "SELECT * FROM `usuarios` WHERE `Status` >= 1  and Id_Usuario != 28 and Id_Usuario != 27 and Id_Usuario != 29 and Id_Usuario != 122 ";
        $stmt = $this->_dbh->prepare($sql);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos de usuarios';
        }

        return $response;
    }

    public function obtenerVendedoresActivosPorSucursalSecuencia($idSucursal)
    {
        $sql = "SELECT * FROM `usuarios` WHERE (`Tipo` = 4 OR `Tipo` = 2) AND `Status` = 1 AND `Id_Sucursal` = ?  or (Id_Usuario = 9 || Id_Usuario = 63)";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idSucursal, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['vendedores'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener la lista '
                . 'de vendedores. ' . TICKETS;
        }

        return $response;
    }

    public function obtenerIdUsuariosBloqueadosPorSucursal($idSucursal)
    {
        $sql = "SELECT u.Id_Usuario, u.Nombre, u.Apellido_Paterno,u.Apellido_Materno FROM bloqueos_usuarios as bu 
             inner join usuarios as u on u.Id_Usuario = bu.id_usuario 
             WHERE bu.desbloqueo = 0 and u.Id_Sucursal = ? 
             GROUP BY u.Id_Usuario ORDER BY u.Usuario ASC";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idSucursal, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos de la solicitud. ' . TICKETS;
        }

        return $response;
    }

    public function desbloqueaUsuarioGerente($id_bloqueos_usuarios)
    {
        $sql = "UPDATE `bloqueos_usuarios` SET `desbloqueo` = 1 WHERE `id_usuario` = ? and `desbloqueo` = 0  ";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $id_bloqueos_usuarios, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Ocurrió un error al desbloquear al usuario. ' . TICKETS;
        }

        return $response;
    }

    public function obtenermotivosUsuarioBloqueadoGerente($idUsuario)
    {
        $sql = " select * from bloqueos_usuarios  "
            . " where id_usuario = ? and desbloqueo = 0  ";
        $stmt = $this->_dbh->prepare($sql);

        $stmt->bindValue(1, $idUsuario, PDO::PARAM_INT);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los datos de la solicitud. ' . TICKETS;
        }

        return $response;
    }

    public function obtenerRetardos($idUsuario, $fechainicial, $fechaFinal)
    {
        $sql = "select * from asistencia where Id_Usuario = ? and fecha between ? and ? and Asistencia = 2 ";
        $stmt = $this->_dbh->prepare($sql);

        $stmt->bindValue(1, $idUsuario, PDO::PARAM_INT);
        $stmt->bindValue(2, $fechainicial, PDO::PARAM_STR);
        $stmt->bindValue(3, $fechaFinal, PDO::PARAM_STR);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['total'] = $stmt->rowCount();
            // $response['datos'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), __FILE__, __LINE__);
            $response['status'] = false;
            $response['error'] = 'Lo siento, no pude obtener los dias injustificados. '
                . TICKETS;
        }

        return $response;
    }

    public function obtenerVendedoresForecastActivosPorSucursal($idSucursal, $fecha)
    {
        $sql = "SELECT u.* FROM usuarios as u 
inner JOIN forecast_mensual as fm on fm.Id_Usuario = u.Id_Usuario
WHERE (u.Tipo = 4 OR u.Tipo = 2) AND u.Status = 1 AND fm.Id_Sucursal = ? and fm.Fecha = ? GROUP by u.Id_Usuario";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idSucursal, PDO::PARAM_INT);
        $stmt->bindValue(2, $fecha, PDO::PARAM_STR);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['vendedores'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener la lista '
                . 'de vendedores. ' . TICKETS;
        }

        return $response;
    }

    public function obtenerUsuarioPorUsuario(Usuario $u)
    {
        $sql = "SELECT * FROM `usuarios` WHERE `Usuario` = ? and Status = 1";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $u->__get('_usuario'), PDO::PARAM_STR);
        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetch(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), __FILE__, __LINE__);
            $response['status'] = false;
            $response['error'] = 'Lo siento, no pude validar los datos de usuario. '
                . TICKETS;
        }

        return $response;
    }

    public function obtenerUsuariosActivosConLicencia($fecha)
    {
        $sql = "SELECT * FROM `usuarios` WHERE `Status` >= 1 and Vigencia_Licencia >= ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $fecha, PDO::PARAM_STR);

        try {
            $stmt->execute();
            $response['status'] = true;
            $response['usuarios'] = $stmt->fetchAll(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener la lista '
                . 'de usuarios. ' . TICKETS;
        }

        return $response;
    }

    public function guardarLicencia(Usuario $u)
    {
        $sql = "UPDATE `usuarios` SET `Licencia` = ?, `Vigencia_Licencia` = ? WHERE `Id_Usuario` = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $u->__get('_numLicencia'), PDO::PARAM_STR);
        $stmt->bindValue(2, $u->__get('_vigenciaLicencia'), PDO::PARAM_STR);
        $stmt->bindValue(3, $u->__get('_idUsuario'), PDO::PARAM_INT);

        try {
            $stmt->execute();
            $response['status'] = true;
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Ocurrió un error al actualizar la asistencia. ' . TICKETS;
        }

        return $response;
    }
    
        public function guardaRegimenFiscal(Usuario $regimenFiscal, $idUser){
        Utils::logInfo("INSERT REGIMEN", "", "");
        $SQL = "INSERT INTO `regimen_fiscal_user` (`id_regimen`, `regimen_fiscal`, `id_user`) VALUES (NULL, ?, ?) ";

        $stmt = $this->_dbh->prepare($SQL);
        $stmt->bindValue(1, $regimenFiscal->__get('_nombreRegi'), PDO::PARAM_STR);
        $stmt->bindValue(2, $regimenFiscal->__get('_id_user'), PDO::PARAM_INT);
        try{
            $stmt->execute();
            $response['status'] = true;
            $response['idRegimen'] = $idUser;
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al guardar el regimen fiscal. '
                . TICKETS;
        }

    }

    public function getRegimenFiscal($idUser, $regimenFinal){
        Utils::logInfo("LLEGA A GET REGIMEN", "", "");
        $sql = "SELECT `regimen_fiscal` FROM `regimen_fiscal_user` WHERE `id_user` = ? AND `regimen_fiscal` = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idUser, PDO::PARAM_INT);
        $stmt->bindValue(2, $regimenFinal, PDO::PARAM_STR);

        try {
            $stmt->execute();
            $response['status'] = true;
            $response['regimenF'] = $stmt->fetchAll(PDO::FETCH_OBJ);
            $response['total'] = $this->_dbh->query('SELECT FOUND_ROWS();')->fetch(PDO::FETCH_COLUMN);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los regimenes del usuario. '
                . TICKETS;
        }

        return $response;
    }

    public function getRegimenesFiscales($idUser){
        $sql = "SELECT * FROM `regimen_fiscal_user` WHERE `id_user` = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idUser, PDO::PARAM_INT);

        try {
            $stmt->execute();
            $response['status'] = true;
            $response['regimenFiscal'] = $stmt->fetchAll(PDO::FETCH_OBJ);
            $response['total'] = $this->_dbh->query('SELECT FOUND_ROWS();')->fetch(PDO::FETCH_COLUMN);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, ocurrió un error al obtener los regimenes. '
                . TICKETS;
        }

        return $response;
    }
    
    
    public function obtenerDireccionUsuario($idUsuario){
        $sql = "SELECT * FROM `usuarios` INNER JOIN `direcciones` ON `direcciones`.`Id_Direccion` = `usuarios`.`Id_Direccion` WHERE `Id_Usuario` = ?";
        $stmt = $this->_dbh->prepare($sql);
        $stmt->bindValue(1, $idUsuario, PDO::PARAM_INT);
        try{
            $stmt->execute();
            $response['status'] = true;
            $response['usuario'] = $stmt->fetch(PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            Utils::log($e->getMessage(), $e->getFile(), $e->getLine());
            $response['status'] = false;
            $response['error'] = 'Lo siento, no pude obtener los datos del Usuario. '
                . TICKETS;
        }

        return $response;
    }

}
